--[[----------------------------------------------------------------------------

Task.lua
Upload photos to Lumethic API

------------------------------------------------------------------------------]]

-- Lightroom API
local LrPathUtils = import 'LrPathUtils'
local LrFileUtils = import 'LrFileUtils'
local LrErrors = import 'LrErrors'
local LrDialogs = import 'LrDialogs'
local LrHttp = import 'LrHttp'
local LrTasks = import 'LrTasks'

--============================================================================--

Task = {}

--------------------------------------------------------------------------------

-- Configuration constants
local MAX_RETRIES = 3
local RETRY_DELAY = 2 -- seconds
local TIMEOUT_SECONDS = 30

-- Error categories for better error handling
local ErrorCategory = {
    NETWORK = 'network',
    AUTH = 'auth',
    FILE = 'file',
    SERVER = 'server',
    VALIDATION = 'validation'
}

local function categorizeError(statusCode, errorMessage)
    -- Convert statusCode to number to ensure safe arithmetic operations
    local status = tonumber(statusCode)
    
    if not status then
        return ErrorCategory.NETWORK, LOC '$$$/Lumethic/Errors/NetworkError=Network error occurred. Please check your connection.'
    end
    
    if status == 401 or status == 403 then
        return ErrorCategory.AUTH, LOC '$$$/Lumethic/Errors/AuthenticationFailed=Authentication failed. Please verify your API key.'
    elseif status == 404 then
        return ErrorCategory.SERVER, LOC '$$$/Lumethic/Errors/ConnectionFailed=Failed to connect to the server. Please check the API URL and your internet connection.'
    elseif status >= 400 and status < 500 then
        return ErrorCategory.VALIDATION, LOC('$$$/Lumethic/Errors/ServerError=Server error (HTTP ^1). Please try again later.', status)
    elseif status >= 500 then
        return ErrorCategory.SERVER, LOC('$$$/Lumethic/Errors/ServerError=Server error (HTTP ^1). Please try again later.', status)
    else
        return ErrorCategory.NETWORK, errorMessage or LOC '$$$/Lumethic/Errors/NetworkError=Network error occurred. Please check your connection.'
    end
end

local function validateFile(filePath, fileType)
    if not filePath or filePath == '' then
        return false, LOC('$$$/Lumethic/Errors/FileNotFound=Source file not found: ^1', fileType or 'unknown')
    end
    
    -- Only check file existence if LrFileUtils.exists is available
    if LrFileUtils and LrFileUtils.exists and not LrFileUtils.exists(filePath) then
        return false, LOC('$$$/Lumethic/Errors/FileNotFound=Source file not found: ^1', filePath)
    end
    
    return true, nil
end

local function readFileWithErrorHandling(filePath, fileType)
    local success, error = validateFile(filePath, fileType)
    if not success then
        return nil, error
    end
    
    local handle, openError = io.open(filePath, 'rb')
    if not handle then
        return nil, LOC('$$$/Lumethic/Errors/FileReadError=Unable to read file: ^1', filePath)
    end
    
    local content = handle:read('*all')
    handle:close()
    
    if not content then
        return nil, LOC('$$$/Lumethic/Errors/FileReadError=Unable to read file: ^1', filePath)
    end
    
    return content, nil
end

local function uploadToApiWithRetry(exportParams, renderedJpegPath, sourceRawPath, progressScope, currentPhotoNum, nPhotos, attempt)
    attempt = tonumber(attempt) or 1
    
    if attempt > 1 then
        LrTasks.sleep(RETRY_DELAY)
    end
    
    local headers = {
        { field = 'X-API-Key', value = exportParams.apiKey },
    }

    local jpegContent, jpegError = readFileWithErrorHandling(renderedJpegPath, 'JPEG')
    if not jpegContent then
        local errorMsg
        local fileReadErrorKey = "$$$/Lumethic/Errors/FileReadError=Unable to read file: ^1"
        if LOC and LOC ~= nil and type(LOC) == "function" then
            errorMsg = LOC(fileReadErrorKey, renderedJpegPath)
        else
            errorMsg = "Unable to read file: " .. renderedJpegPath -- Fallback
        end
        if LrDialogs and LrDialogs.showError then LrDialogs.showError(errorMsg .. "\n" .. (jpegError or "")) end
        return nil, errorMsg -- Propagate the error message, matching expected return for pre-HTTP errors
    end

    local rawContent, rawError = readFileWithErrorHandling(sourceRawPath, 'RAW')
    if not rawContent then
        local errorMsg
        local fileReadErrorKey = "$$$/Lumethic/Errors/FileReadError=Unable to read file: ^1"
        if LOC and LOC ~= nil and type(LOC) == "function" then
            errorMsg = LOC(fileReadErrorKey, sourceRawPath)
        else
            errorMsg = "Unable to read file: " .. sourceRawPath -- Fallback
        end
        if LrDialogs and LrDialogs.showError then LrDialogs.showError(errorMsg .. "\n" .. (rawError or "")) end
        return nil, errorMsg -- Propagate the error message
    end

    -- Prepare multipart content using LrHttp.postMultipart format
    local content = {
        {
            name = "raw",
            fileName = LrPathUtils.leafName(sourceRawPath),
            value = rawContent,
            contentType = "application/octet-stream"
        },
        {
            name = "image", 
            fileName = LrPathUtils.leafName(renderedJpegPath),
            value = jpegContent,
            contentType = "image/jpeg"
        }
    }

    -- Make the HTTP request using postMultipart for cleaner multipart handling
    local result, hdrs = LrHttp.postMultipart(exportParams.apiUrl .. '/v1/verifications/', content, headers, TIMEOUT_SECONDS)
    
    -- Handle network errors
    if result == nil then
        local category, message = categorizeError(nil, LOC "$$$/Lumethic/Errors/NetworkTimeoutOrConnectionError=Network timeout or connection error")
        
        -- Retry for network errors if we haven't exceeded max retries
        if attempt < MAX_RETRIES and category == ErrorCategory.NETWORK then
            return uploadToApiWithRetry(exportParams, renderedJpegPath, sourceRawPath, progressScope, currentPhotoNum, nPhotos, attempt + 1)
        end
        
        return false, message, category
    end
    
    -- Handle HTTP errors with safe arithmetic operations
    local statusCode = tonumber(hdrs.status) or 0
    if statusCode < 200 or statusCode >= 300 then
        local category, message = categorizeError(statusCode) -- message is the LOC'd string
        
        -- Retry for server errors (5xx) if we haven't exceeded max retries
        if attempt < MAX_RETRIES and category == ErrorCategory.SERVER and statusCode >= 500 then
            return uploadToApiWithRetry(exportParams, renderedJpegPath, sourceRawPath, progressScope, currentPhotoNum, nPhotos, attempt + 1)
        end
        
        -- If retries are exhausted or it's a non-retryable HTTP error, show the error
        if LrDialogs and LrDialogs.showError then
            LrDialogs.showError(message) 
        end
        
        return false, message, category
    end
    
    -- Upload successful - return immediately without any progress updates
    return true, nil, nil
end
Task.uploadToApiWithRetry = uploadToApiWithRetry -- Exposed for testing

local function validateExportParams(exportParams)
    -- Validate API Key
    if not exportParams.apiKey or exportParams.apiKey == '' then
        LrErrors.throwUserError(LOC '$$$/Lumethic/Errors/EmptyApiKey=API key is required for authentication.')
    end
    
    -- Validate API URL
    if not exportParams.apiUrl or exportParams.apiUrl == '' then
        LrErrors.throwUserError(LOC '$$$/Lumethic/Errors/EmptyApiUrl=API URL is required.')
    end
    
    -- Validate URL format
    if not string.match(exportParams.apiUrl, '^https?://') then
        LrErrors.throwUserError(LOC '$$$/Lumethic/Errors/InvalidApiUrl=Invalid API URL. Please enter a valid URL starting with http:// or https://')
    end
end

function Task.processRenderedPhotos(functionContext, exportContext)
    -- Make a local reference to the export parameters.
    local exportSession = exportContext.exportSession
    local exportParams = exportContext.propertyTable
    
    -- Validate required parameters
    validateExportParams(exportParams)
    
    -- Set progress title.
    local nPhotosRaw = exportSession:countRenditions()
    local nPhotos = tonumber(nPhotosRaw) or 0

    local progressScope = exportContext:configureProgress {
        title = nPhotos > 1
            and LOC('$$$/Lumethic/Upload/Progress=Uploading ^1 photos to Lumethic', nPhotos)
            or LOC '$$$/Lumethic/Upload/Progress/One=Uploading one photo to Lumethic',
    }
    
    -- Show initial progress message
    if progressScope and progressScope.setCaption then
        progressScope:setCaption(LOC '$$$/Lumethic/Upload/Progress/Initializing=Initializing upload process...')
    end
    
    -- Iterate through photo renditions.
    local failures = {}
    local successes = 0
    local processed = 0
    
    for i, rendition in exportContext:renditions{ stopIfCanceled = true } do
        -- Update progress for current photo (before render) - similar to Flickr plugin
        if progressScope and progressScope.setPortionComplete then
            progressScope:setPortionComplete( ( i - 1 ) / nPhotos )
        end
        
        -- Update progress caption for current photo
        if progressScope and progressScope.setCaption then
            progressScope:setCaption(LOC('$$$/Lumethic/Upload/Progress/Processing=Processing photo ^1 of ^2...', i, nPhotos))
        end
        
        -- Wait for next photo to render.
        local success, renderedJpegPath = rendition:waitForRender()
        
        -- Update progress after render but before upload - similar to Flickr plugin
        if progressScope and progressScope.setPortionComplete then
            progressScope:setPortionComplete( ( i - 0.5 ) / nPhotos )
        end
        
        -- Check for cancellation again after photo has been rendered.
        if progressScope:isCanceled() then 
            if progressScope.done then
                progressScope:done()
            end
            return
        end
        
        if success then
            local filename = LrPathUtils.leafName(renderedJpegPath)
            local sourceRawPath = rendition.photo:getRawMetadata('path')
            
            -- Validate that we have a RAW file
            if not sourceRawPath or sourceRawPath == '' then
                table.insert(failures, {
                    filename = filename,
                    error = LOC('$$$/Lumethic/Errors/RawFileNotFound=RAW file not found for photo: ^1', filename),
                    category = ErrorCategory.FILE
                })
            else
                -- Show upload starting progress
                if progressScope and progressScope.setCaption then
                    progressScope:setCaption(LOC('$$$/Lumethic/Upload/Progress/Uploading=Uploading photo ^1 of ^2...', i, nPhotos))
                end
                
                -- Upload both RAW and rendered JPEG files (this blocks until HTTP POST is completely finished)
                local uploadSuccess, errorMessage, errorCategory = uploadToApiWithRetry(exportParams, renderedJpegPath, sourceRawPath, progressScope, i, nPhotos)
                
                -- Process upload result AFTER upload is completely finished
                if uploadSuccess then
                    successes = successes + 1
                    -- Show success message for this photo
                    if progressScope and progressScope.setCaption then
                        progressScope:setCaption(LOC('$$$/Lumethic/Upload/Progress/PhotoSuccess=Successfully uploaded photo ^1 of ^2.', i, nPhotos))
                    end
                else
                    table.insert(failures, {
                        filename = filename,
                        error = errorMessage,
                        category = errorCategory
                    })
                    -- Show error message for this photo
                    if progressScope and progressScope.setCaption then
                        progressScope:setCaption(LOC('$$$/Lumethic/Upload/Progress/PhotoFailed=Failed to upload photo ^1 of ^2.', i, nPhotos))
                    end
                end
            end
            
            -- When done with photo, delete temp file.
            LrFileUtils.delete(renderedJpegPath)
        else
            table.insert(failures, {
                filename = LOC '$$$/Lumethic/Errors/UnknownFile=Unknown file',
                error = LOC '$$$/Lumethic/Errors/RenderFailed=Failed to render photo',
                category = ErrorCategory.FILE
            })
        end
        
        -- After this photo is completely processed, progress will naturally advance 
        -- to the next iteration (which updates to i/nPhotos at the start of the next loop)
        -- Small delay to make progress visible
        LrTasks.sleep(0.1)
    end
    
    -- Set final progress message but DON'T complete progress yet
    if progressScope and progressScope.setCaption then
        if #failures == 0 then
            progressScope:setCaption(LOC '$$$/Lumethic/Upload/Caption/AllSuccess=Upload complete: All successful.')
        elseif successes > 0 then
            progressScope:setCaption(LOC '$$$/Lumethic/Upload/Caption/PartialSuccess=Upload complete: Some errors.')
        else
            progressScope:setCaption(LOC '$$$/Lumethic/Upload/Caption/AllFailed=Upload failed.')
        end
    end
    
    -- Show completion messages BEFORE completing progress
    if #failures == 0 then
        -- All successful
        local message = nPhotos > 1 
            and LOC '$$$/Lumethic/Success/UploadComplete=All photos uploaded successfully!'
            or LOC '$$$/Lumethic/Success/SingleUploadComplete=Photo uploaded successfully!'
        LrDialogs.message(LOC '$$$/Lumethic/Dialog/Title/UploadComplete=Upload Complete', message, 'info')
    elseif successes > 0 then
        -- Partial success
        local message = LOC '$$$/Lumethic/Warnings/PartialSuccess=Upload completed with some errors. Check the error log for details.'
        local details = string.format(LOC "$$$/Lumethic/Dialog/Details/PartialSuccessHeader=%1 of %2 photos uploaded successfully.\\n\\nFailures:", successes, nPhotos)
        for _, failure in ipairs(failures) do
            details = details .. '\\n• ' .. failure.filename .. ': ' .. failure.error
        end
        LrDialogs.message(LOC '$$$/Lumethic/Dialog/Title/PartialSuccess=Upload Partially Successful', message, 'warning')
    else
        -- All failed
        local message
        if #failures == 1 then
            message = LOC '$$$/Lumethic/Upload/Errors/OneFileFailed=1 file failed to upload correctly.'
        else
            message = LOC('$$$/Lumethic/Upload/Errors/SomeFileFailed=^1 files failed to upload correctly.', #failures)
        end
        
        local details = LOC "$$$/Lumethic/Dialog/Details/AllFailedHeader=Upload errors:\\n"
        for _, failure in ipairs(failures) do
            details = details .. '\\n• ' .. failure.filename .. ': ' .. failure.error
        end
        
        LrDialogs.message(LOC '$$$/Lumethic/Dialog/Title/AllFailed=Upload Failed', message, 'critical')
    end
    
    -- Now set progress to 100% and complete it together
    if progressScope and progressScope.setPortionComplete then
        progressScope:setPortionComplete(1.0)
    end
    
    -- Complete the progress ONLY after the user has seen the completion dialog
    if progressScope and progressScope.done then
        progressScope:done()
    end
end