--[[----------------------------------------------------------------------------

PluginInfoProvider.lua
Plugin Manager dialog for Lightroom Lumethic plugin

------------------------------------------------------------------------------]]

local LrView = import 'LrView'
local LrPrefs = import 'LrPrefs'

local prefs = LrPrefs.prefsForPlugin()

local DEFAULT_API_URL = 'https://api.lumethic.com'

return {
    
    startDialog = function(propertyTable)
        -- Initialize from saved preferences, with defaults
        -- Always ensure we have a valid API URL
        local savedUrl = prefs.apiUrl
        if not savedUrl or savedUrl == '' then
            savedUrl = DEFAULT_API_URL
            prefs.apiUrl = savedUrl
        end
        propertyTable.apiUrl = savedUrl
        propertyTable.apiKey = prefs.apiKey or ''
        
        -- Add observers to sync changes to prefs
        propertyTable:addObserver('apiUrl', function(props, key, newValue)
            prefs.apiUrl = newValue
        end)
        
        propertyTable:addObserver('apiKey', function(props, key, newValue)
            prefs.apiKey = newValue
        end)
    end,
    
    endDialog = function(propertyTable)
        -- Save settings to preferences when dialog closes
        prefs.apiUrl = propertyTable.apiUrl
        prefs.apiKey = propertyTable.apiKey
    end,
    
    sectionsForTopOfDialog = function(f, propertyTable)
        local bind = LrView.bind
        local share = LrView.share
        
        return {
            {
                title = LOC '$$$/Lumethic/PluginManager/Settings=Lumethic API Settings',
                synopsis = bind { key = 'apiUrl', object = propertyTable },
                
                f:row {
                    f:static_text {
                        title = LOC '$$$/Lumethic/ExportDialog/ApiUrl=API URL:',
                        alignment = 'right',
                        width = share 'pluginLabelWidth'
                    },
                    
                    f:edit_field {
                        value = bind 'apiUrl',
                        immediate = true,
                        fill_horizontal = 1,
                        tooltip = LOC "$$$/Lumethic/ExportDialog/Tooltips/ApiUrl=Enter the Lumethic API URL (e.g., https://api.example.com)",
                        placeholder_string = DEFAULT_API_URL,
                    },
                },
                
                f:row {
                    f:static_text {
                        title = LOC '$$$/Lumethic/ExportDialog/ApiKey=API Key:',
                        alignment = 'right',
                        width = share 'pluginLabelWidth'
                    },
                    
                    f:edit_field {
                        value = bind 'apiKey',
                        immediate = true,
                        fill_horizontal = 1,
                        password = true,
                        tooltip = LOC "$$$/Lumethic/ExportDialog/Tooltips/ApiKey=Enter your Lumethic API key",
                    },
                },
                
                f:row {
                    f:spacer {
                        width = share 'pluginLabelWidth'
                    },
                    
                    f:static_text {
                        title = LOC '$$$/Lumethic/PluginManager/Hint=These settings will be used as defaults for new exports.',
                        fill_horizontal = 1,
                    },
                },
            },
        }
    end,
    
}
