--[[----------------------------------------------------------------------------

ExportDialogSections.lua
Export dialog customization for Lightroom Lumethic uploader

------------------------------------------------------------------------------]]

-- Lightroom SDK
local LrView = import 'LrView'
local LrBinding = import 'LrBinding'
local LrPrefs = import 'LrPrefs'

--============================================================================--

ExportDialogSections = {}

-- Get plugin preferences for syncing with Plugin Manager settings
local prefs = LrPrefs.prefsForPlugin()

-------------------------------------------------------------------------------

local function validateApiUrl(url)
    if not url or url == '' then
        return false, LOC '$$$/Lumethic/Errors/EmptyApiUrl=API URL is required.'
    end
    
    -- Check if URL starts with http:// or https://
    if not string.match(url, '^https?://') then
        return false, LOC '$$$/Lumethic/Errors/InvalidApiUrl=Invalid API URL. Please enter a valid URL starting with http:// or https://'
    end
    
    return true, nil
end

local function validateApiKey(key)
    if not key or key == '' then
        return false, LOC '$$$/Lumethic/ExportDialog/Messages/EnterApiKey=Enter your API key'
    end
    
    return true, nil
end

local function updateExportStatus(propertyTable)
    local message = nil
    local hasError = false
    
    repeat
        -- Validate API Key first (most critical)
        local keyValid, keyError = validateApiKey(propertyTable.apiKey)
        if not keyValid then
            message = keyError
            hasError = true
            break
        end
        
        -- Validate API URL (required for export)
        local urlValid, urlError = validateApiUrl(propertyTable.apiUrl)
        if not urlValid then
            message = urlError
            hasError = true
            break
        end
        
    until true
    
    if hasError then
        propertyTable.message = message
        propertyTable.hasError = true
        propertyTable.hasNoError = false
        propertyTable.LR_cantExportBecause = message
    else
        propertyTable.message = nil
        propertyTable.hasError = false
        propertyTable.hasNoError = true
        propertyTable.LR_cantExportBecause = nil
    end
end

-------------------------------------------------------------------------------

function ExportDialogSections.startDialog(propertyTable)
    -- Sync from plugin preferences if export settings are empty/default
    -- This ensures Plugin Manager settings are used as defaults
    if (not propertyTable.apiUrl or propertyTable.apiUrl == '' or propertyTable.apiUrl == 'https://api.lumethic.com') and prefs.apiUrl and prefs.apiUrl ~= '' then
        propertyTable.apiUrl = prefs.apiUrl
    end
    if (not propertyTable.apiKey or propertyTable.apiKey == '') and prefs.apiKey and prefs.apiKey ~= '' then
        propertyTable.apiKey = prefs.apiKey
    end
    
    -- Add observers for both API URL and API Key
    -- These observers also sync changes back to plugin preferences
    propertyTable:addObserver('apiUrl', function(props, key, newValue)
        updateExportStatus(props)
        -- Sync to plugin preferences
        prefs.apiUrl = newValue
    end)
    
    propertyTable:addObserver('apiKey', function(props, key, newValue)
        updateExportStatus(props)
        -- Sync to plugin preferences
        prefs.apiKey = newValue
    end)
    
    -- Initial validation
    updateExportStatus(propertyTable)
end

-------------------------------------------------------------------------------

function ExportDialogSections.sectionsForBottomOfDialog(f, propertyTable)
    local bind = LrView.bind
    local share = LrView.share
    
    local result = {
        {
            title = LOC '$$$/Lumethic/ExportDialog/Settings=Lumethic Settings',
            
            f:row {
                f:static_text {
                    title = LOC '$$$/Lumethic/ExportDialog/ApiUrl=API URL:',
                    alignment = 'right',
                    width = share 'labelWidth'
                },
                
                f:edit_field {
                    value = bind 'apiUrl',
                    immediate = true,
                    fill_horizontal = 1,
                    tooltip = LOC "$$$/Lumethic/ExportDialog/Tooltips/ApiUrl=Enter the Lumethic API URL (e.g., https://api.example.com)",
                },
            },
            
            f:row {
                f:static_text {
                    title = LOC '$$$/Lumethic/ExportDialog/ApiKey=API Key:',
                    alignment = 'right',
                    width = share 'labelWidth'
                },
                
                f:edit_field {
                    value = bind 'apiKey',
                    immediate = true,
                    fill_horizontal = 1,
                    password = true,  -- Hide the API key for security
                    tooltip = LOC "$$$/Lumethic/ExportDialog/Tooltips/ApiKey=Enter your Lumethic API key",
                },
            },
            
            f:row {
                f:spacer {
                    width = share 'labelWidth'
                },
                
                f:static_text {
                    title = bind 'message',
                    fill_horizontal = 1,
                    visible = bind 'hasError',
                },
            },
        },
    }
    
    return result
end